/*
 Copyright 2014 Adobe Systems Incorporated.  All rights reserved.

Purpose-
This file has the implementation of Selectors Editing functionality in Live Edit view
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global $, liveViewObject, dwObject, liveExtensionsConfigPath, dwExtensionController, DW_EXTENSION_EVENT*/

/* Constants used in this file */
var CONSTANTS = {
    ExtensionID: "editableSelectors",
    DwDivTagName: "dw-div",
    EscapeKeyCode: 27,
    EnterKeyCode: 13,
    UpArrowKeyCode: 38,
    RightArrowKeyCode: 39,
    DownArrowKeyCode: 40,
    OverlayBorderWidth: 1,
    SuggestionsListMaxHeight: 100,
    ExtensionMinHeight: 130,
    ImgTagName: 'img',
    hudEntryID: 'dwLeAuxHudEntry',
    ExtensionPath: '/EditableSelectors',
    Assets: '/assets',
    eshShown: "eshShown",
    endActionCategory: "endAction",
    startAction: "startAction",
    textHudShown: "textHudShown",
    eshAddSelector: "eshAddSelector",
    HudSandWich: "/sandwich.SVG"
};

var DWLE_IMG_CLASS = {
    ImgHudEntry: "dw_imgPropHud",
    ImgSelection: "dw_imgSelection",
    Separator: " "
};

var SELECTOR_CONSTANTS = {
    ElementHudClass: "elementHud",
    ElementHudEditableBgColor: "elementHudEditableBackground",
    ElementHudNonEditableBgColor: "elementHudNonEditableBackground",
    TagnameClass: "tagName",
    OverlayClass: "overlay",
    OverlayEditableBorderColor: "overlayEditableBorderColor",
    OverlayNonEditableBorderColor: "overlayNonEditableBorderColor",
    EditableSelectorClass: "selector",
    NonEditableSelectorClass: "selectorNonEditable",
    ExpandedSelectorClass: "selectorExpanded",
    SelectorsContainerClass: "selectorsContainer",
    SelectorNameClass: "selectorName",
    RemoveSelectorButtonClass: "removeSelectorButton",
    RemoveSelectorButtonString: "x",
    AddNewSelectorButtonID: "addNewSelectorButton",
    NewSelectorTextInputID: "newSelectorTextInput",
    NewSelectorTextInputJQueryID: "#newSelectorTextInput",
    AddNewSelectorButtonString: "+"
};

var DW_ESH_HEADLIGHTS = {
    ELV_ESH: "ELV ESH",
    OTH_ELV: "OTH ELV",
    ESH_ADD: "ESH Add",
    ESH_DELETE: "ESH Delete",
    ESH_MULTIADD: "ESH MultiAdd",
    ESH_ESCAPE: "ESH Escape"
};

/* Object definition */

function EditableSelectors() {
    'use strict';
}

/*
function:initialize - initialize variabes, add event listeners

Arguments: None

Return : None
*/
EditableSelectors.prototype.initialize = function () {
    'use strict';
        
    //initialize variables
    this.m_documentClassesForElement = null;
    this.m_documentIdForElement = null;
    this.m_liveClassesForElement = null;
    this.m_liveIdForElement = null;
    this.m_currentSelectedElement = null;
    this.m_hudIsOnTop = true;
    this.textEditing = false;
    this.m_initiateTextEditOnKeyUp = false;
    
    //add listener for messages from Live View
    window.addEventListener("message", this.messageHandler.bind(this), false);
    
    //initialize our container structure
    this.m_hudContainer = document.body;
    this.m_hudContentDiv = document.createElement(CONSTANTS.DwDivTagName);
    this.m_hudContentDiv.setAttribute('class', SELECTOR_CONSTANTS.ElementHudClass);
    this.m_hudContainer.appendChild(this.m_hudContentDiv);
    
    //create overlay component
    this.m_overlayDiv = document.createElement(CONSTANTS.DwDivTagName);
    this.m_overlayDiv.setAttribute('class', SELECTOR_CONSTANTS.OverlayClass);
    this.m_hudContainer.appendChild(this.m_overlayDiv);
    
    //set click handler
    this.m_hudContentDiv.onmouseup = this.ClickHandler.bind(this);
    
    //add key up handler
    var keyUpHandler = function (evt) {
        evt = evt || window.event;
        if (evt.keyCode === CONSTANTS.EnterKeyCode) {
            if (this.m_initiateTextEditOnKeyUp && this.globalController && this.m_currentSelectedElement) {
                this.globalController.enterIntoTextEditing(this.m_currentSelectedElement);
            }
            this.m_initiateTextEditOnKeyUp = false;
        } else if (evt.keyCode === CONSTANTS.DownArrowKeyCode || evt.keyCode === CONSTANTS.UpArrowKeyCode) {
            //In new selector input field, cursor moves to the beginning while navigating through suggestions
            //This is the behaviour for the jQuery plugin with contentEditable input field. so we have
            //to put the cursor at the end 
            var inputElement = document.getElementById(SELECTOR_CONSTANTS.NewSelectorTextInputID);
            //check the target node
            if (inputElement && evt.target === inputElement) {
                var inputStr = inputElement.innerHTML;
                if (inputStr && inputElement.childNodes[0]) {
                    //set the document selection to bring the cursor to the end
                    var range = document.createRange();
                    var sel = window.getSelection();
                    range.setStart(inputElement.childNodes[0], inputStr.length);
                    range.collapse(true);
                    sel.removeAllRanges();
                    sel.addRange(range);
                }
                //on MAC, first keypress is lost for some reason after a up/down press
                //so send a dummy right arrow key press to get rid of this
                //this is a noop as the cursor is already at the right most position in the input
                $(SELECTOR_CONSTANTS.NewSelectorTextInputJQueryID).focus().trigger({type: "keypress", which: CONSTANTS.RightArrowKeyCode});
            }
        }
    };
    
    //add key down handler to handle escape and enter key
    var keyDownHandler = function (evt) {
        evt = evt || window.event;
        if (evt.keyCode === CONSTANTS.EscapeKeyCode) {
            this.escapeKeyPressed();
        } else if (evt.keyCode === CONSTANTS.EnterKeyCode) {
            //pressing enter when HUD on, should enter text editing
            evt.stopPropagation();
            if (this.m_currentSelectedElement && liveViewObject.isExtensionVisible(CONSTANTS.ExtensionID)) {
                //we should initiate text editing only on keyup, 
                //otherwise keyup will be handled by text editing session 
                this.m_initiateTextEditOnKeyUp = true;
            }
        }
    };
    
    window.addEventListener("keyup", keyUpHandler.bind(this), false);
    window.addEventListener("keydown", keyDownHandler.bind(this), false);
    //we should handle escape on parent doc also
    if (parent) {
        parent.addEventListener("keydown", keyDownHandler.bind(this), false);
        parent.addEventListener("keyup", keyUpHandler.bind(this), false);
        //add resize handler too
        parent.addEventListener("resize", this.windowResized.bind(this), false);
    }
    
    dwExtensionController.addListener(DW_EXTENSION_EVENT.AUX_HUD_ESCAPED, keyDownHandler, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.AUX_HUD_HIDDEN, this.reload, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.RELOAD_ESH, this.reload, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.ELEMENT_DIMENSION_CHANGED, this.repositionHUD, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.TEXT_EDIT_BEGIN, this.textEditBegin, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.TEXT_EDIT_END, this.textEditEnd, this);
    
    this.globalController = window.parent.globalController;
    this.locStrings = {};
    this.locStrings.auxHudTooltip = this.globalController.getLocalizedString('ESH_SWIcon_ToolTip');
    this.locStrings.addTooltip = this.globalController.getLocalizedString('ESH_AddButton_Tooltip');
    this.locStrings.removeTooltip = this.globalController.getLocalizedString('ESH_RemoveButton_Tooltip');
    
    this.shouldScroll = true;
};

EditableSelectors.prototype.textEditBegin = function () {
    'use strict';
    this.destroy();
    this.textEditing = true;
};

EditableSelectors.prototype.textEditEnd = function () {
    'use strict';
    this.textEditing = false;
};

/*
function:escapeKeyPressed - handler for escape key press

Arguments: None

Return : None
*/
EditableSelectors.prototype.escapeKeyPressed = function () {
    'use strict';
	//if user is adding a new selector, that should be dismissed on escape key
    var inputElem = document.getElementById(SELECTOR_CONSTANTS.NewSelectorTextInputID);
    if (inputElem && inputElem.offsetWidth > 0) {
        this.focusLostFromNewSelectorInput(null, null);
	} else {
		this.destroy(true);
	}
};

/*
function:documentOrWindowResized - handler for document or window resize. We need to recreate HUD

Arguments: None

Return : None
*/
EditableSelectors.prototype.windowResized = function () {
    'use strict';
	if (liveViewObject.isExtensionVisible(CONSTANTS.ExtensionID)) {
        this.createElementHud();
    }
};


EditableSelectors.prototype.isAnyAuxHudVisible = function () {
    'use strict';
    if (this.m_currentSelectedElement) {
        return window.parent.liveViewExtensionsObject.getAuxHudVisibility(this.m_currentSelectedElement.tagName.toLowerCase());
    }
    return false;
};
/*
function:destroy - destroy ourself

Arguments: None

Return : None
*/
EditableSelectors.prototype.destroy = function (isEscaped) {
    'use strict';
    //hide ourself

    if (this.isAnyAuxHudVisible()) {
        if (isEscaped) {
            this.hideAuxHud();
        } else {
            this.hideAuxHud(true);
        }
        return;
    }
    
	liveViewObject.hideExtensionById(CONSTANTS.ExtensionID);
    this.m_currentSelectedElement = null;
    
    //log headlights
    if (isEscaped) {
        dwObject.logHeadlightsData(DW_ESH_HEADLIGHTS.OTH_ELV, DW_ESH_HEADLIGHTS.ESH_ESCAPE);
    }
};

EditableSelectors.prototype.reload = function (e) {
    'use strict';
    
    if (e && typeof e.shouldScroll !== "undefined") {
        this.m_shouldScroll = e.shouldScroll;
    }
    else
        this.m_shouldScroll = true;
    
	//get the current class and id attributes from document
	//this will be returned as a callback
	var callbackFunction = function (resultObj) {
		this.getDocumentElementAttributesCallback(resultObj);
	}.bind(this);
	dwObject.getDWDocumentElementClassAndID(this.m_currentSelectedElement, callbackFunction);
};

/*
function:messageHandler - Handler for messages from Live View

Arguments: None

Return : None
*/
EditableSelectors.prototype.messageHandler = function (e) {
    'use strict';
    if (e.data && e.data.type) {
        if (e.data.type === DW_EXTENSION_EVENT.SELECTION_CHANGE) {
            if (this.textEditing) {
                // don't do anything and return
                return;
            }
            var new_selection = liveViewObject.getCurrentSelectedElement();
            //removing the following check, since it will not update the changes
            //done to the current element (watson bug: 3701306)
            var old_selection = this.m_currentSelectedElement;
            this.m_currentSelectedElement = new_selection;
            if (!this.m_currentSelectedElement || this.m_currentSelectedElement.tagName === "HTML") {
                this.destroy();
                return;
            }
            if (old_selection !== new_selection) {
                this.hideAuxHud(true); // hide before updating the element
            } else {
                //if it is the same element and we are in the middle of adding a selector, do not update
                //as there would not have been any changes done
                var inputElem = document.getElementById(SELECTOR_CONSTANTS.NewSelectorTextInputID);
                if (inputElem && inputElem.offsetWidth > 0) {
                    return;
                }
                this.reload();
            }
        } else if (e.data.type === DW_EXTENSION_EVENT.STYLES_RELOADED) {
            //recreate the HUD if styles changed
            if (liveViewObject.isExtensionVisible(CONSTANTS.ExtensionID)) {
                this.createElementHud();
            }
        } else if (e.data.type === DW_EXTENSION_EVENT.SHOW_AUX_HUD) {
            //this is triggered by cmd+e for now. Logging headlight accordingly
            var sandwichIcon = document.getElementById(CONSTANTS.hudEntryID);
            if (sandwichIcon) {
                this.toggleAuxHudState(true);
            }
        } else if (e.data.type === DW_EXTENSION_EVENT.VIEW_LOST_FOCUS) {
            this.hideAuxHud(true,false,false);
        }
    }
};

/*
function:getDocumentElementAttributesCallback - Callback function from DW with element attributes , class & id
                                                HUD creation starts from here

Arguments: callbackObj.classStr - class attribute value
           callbackObj.idStr - id attribute value

Return : None
*/
EditableSelectors.prototype.getDocumentElementAttributesCallback = function (callbackObj) {
    'use strict';
    //alert('this is callback');
    if (callbackObj) {
        //cleanup cached members
        this.m_documentClassesForElement = null;
        this.m_documentIdForElement = null;
        
        //set the values for document attributes. These values are used to check whether 
        //the curretn applied classes and id for the element in live view is created
        //dynamically or not. If the values are dynamic (not present in document), then 
        //they will not be editable through the HUD
        if (callbackObj.classStr) {
            this.m_documentClassesForElement = callbackObj.classStr.split(' ');
        }
        
        if (callbackObj.idStr) {
            this.m_documentIdForElement = callbackObj.idStr.split(' ');
        }
        
        //now create the HUD
        this.createElementHud();
    }
};

/**
 *  Tells whether an auxiliary HUD exists which can possibly be shown for the tag selected
 *  @param tagName - html tag for which an auxiliary hud is registered
 *  @return BOOLEAN true if an aux hud exist, false if no aux hud is registered
 */
EditableSelectors.prototype.isAuxHudRegisteredForTag = function (tagName) {
    'use strict';
    if (!tagName || typeof tagName !== "string") {
        return false;
    }
    // Need to create a mechanism for extensions to register for invocation
    
    if (liveViewObject.auxiliaryIframeExists[tagName]) {
        return true;
    }
    return false;
};

/**
 *  Generates the HTML to display the sandwich icon that initiates the Auxiliary HUD
 *  @param none
 *  @return STRING outerHTML for the icon to invoke Auxiliary HUD
 */
EditableSelectors.prototype.generateAuxHudIcon = function () {
    'use strict';
    
    var auxHudToggler = document.createElement(CONSTANTS.ImgTagName);
    auxHudToggler.id = CONSTANTS.hudEntryID;
    auxHudToggler.title = this.locStrings.auxHudTooltip;
    auxHudToggler.setAttribute('class', DWLE_IMG_CLASS.ImgHudEntry + DWLE_IMG_CLASS.Separator + DWLE_IMG_CLASS.ImgSelection);
    auxHudToggler.setAttribute('src', liveExtensionsConfigPath + CONSTANTS.ExtensionPath + CONSTANTS.Assets + CONSTANTS.HudSandWich);
    return auxHudToggler.outerHTML;
};

EditableSelectors.prototype.hideAuxHud = function (commit, keepESHOpen, shouldScroll) {
    'use strict';
    if (typeof commit === "undefined") {
        commit = false;
    }
    
    if (typeof keepESHOpen === "undefined") {
        keepESHOpen = false;
    }
    
    if (typeof shouldScroll === "undefined") {
        shouldScroll = true;
    }
    

    var messageDetails = {};
    messageDetails.selectedTag = this.m_currentSelectedElement ? this.m_currentSelectedElement.tagName.toLowerCase() : '*';
    messageDetails.type = DW_EXTENSION_EVENT.HIDE_AUX_HUD;
    messageDetails.commit = commit;
    messageDetails.shouldScroll = shouldScroll;
    
    if (keepESHOpen) {
        messageDetails.keepESHOpen = keepESHOpen;
    }
    dwExtensionController.sendMessage(messageDetails);
};
/**
 *  Toggles the state of the Auxiliary HUD associated with the tag by sending a message to the parent window
 *  Message contains the selected tagName, and the click count for this instance of the sandwich icon
 *  @param none
 *  @param none
 *  @return none
 */

EditableSelectors.prototype.toggleAuxHudState = function (shortcut) {
    'use strict';
    var messageDetails = {};
    messageDetails.selectedTag = this.m_currentSelectedElement.tagName.toLowerCase();
    messageDetails.selectorHudPosition = this.m_hudIsOnTop;
    messageDetails.selectorHudHeight = this.m_hudContentDiv.offsetHeight;
    messageDetails.selectorHudWidth = this.m_hudContentDiv.offsetWidth;
    messageDetails.shortcut = shortcut;
    messageDetails.type = DW_EXTENSION_EVENT.TOGGLE_AUX_HUD;
    dwExtensionController.sendMessage(messageDetails);
};

/*
function:createElementHud - Create and position the HUD

Arguments: None

Return : None
*/
EditableSelectors.prototype.createElementHud = function () {
    'use strict';
	if (this.textEditing) {
        return;
    }
    if (this.m_hudContentDiv && this.m_currentSelectedElement) {
        //get the rect for current element
        var elemRect = liveViewObject.getElementRect(this.m_currentSelectedElement);
        if (elemRect.width <= 0 || elemRect.height <= 0) {
            this.destroy();
            return;
        }
        
        //cleanup cached members
        this.m_liveClassesForElement = null;
        this.m_liveIdForElement = null;

        //get currently applied classes in live view
        var appliedClasses = this.m_currentSelectedElement.getAttribute('class');
        if (appliedClasses) {
            this.m_liveClassesForElement = appliedClasses.split(' ');
        }

        var appliedId = this.m_currentSelectedElement.getAttribute('id');
        if (appliedId) {
            this.m_liveIdForElement = appliedId.split(' ');
        }
        
        //reset current HUD Size so that it will not affect dcumetn size calculations
        liveViewObject.positionExtensionById(CONSTANTS.ExtensionID, 0, 0, 0, 0);
        
        //generate the HUD contents and position
        this.generateHudContents();
        this.positionHud(elemRect);
        
        if( this.m_shouldScroll )
            this.scrollHUDIntoViewIfNeeded();
        
        // Also Reset the flag
        this.m_shouldScroll = true;
    }
};

/*
function:generateHudContents - build the HUD structure, tagName followed by selectors applied and add button

Arguments: None

Return : None
*/
EditableSelectors.prototype.generateHudContents = function () {
    'use strict';
    if (this.m_hudContentDiv && this.m_currentSelectedElement) {
		//get the tagname and add to our HUD structure
		var tagName = this.m_currentSelectedElement.tagName.toLowerCase();
		var tagNameStr = this.createTagWithClassAndContent(CONSTANTS.DwDivTagName,
                                                            SELECTOR_CONSTANTS.TagnameClass,
                                                            tagName);

		//now create the selectors structure
        var contentString = this.buildSelectorElements();
        var auxHudString = '';
        if (this.isAuxHudRegisteredForTag(tagName)) {
            auxHudString = this.generateAuxHudIcon();
        }

		this.m_hudContentDiv.innerHTML = auxHudString + tagNameStr + this.createTagWithClassAndContent(CONSTANTS.DwDivTagName,
                                            SELECTOR_CONSTANTS.SelectorsContainerClass, contentString);
        
        //Set the background color and overlay border color based on editability
        var classesForElementHUD = SELECTOR_CONSTANTS.ElementHudClass + " ";
        var classesForOverlay = SELECTOR_CONSTANTS.OverlayClass + " ";
        if (liveViewObject.isElementEditable(this.m_currentSelectedElement)) {
            classesForElementHUD += SELECTOR_CONSTANTS.ElementHudEditableBgColor;
            classesForOverlay += SELECTOR_CONSTANTS.OverlayEditableBorderColor;
        } else {
            classesForElementHUD += SELECTOR_CONSTANTS.ElementHudNonEditableBgColor;
            classesForOverlay += SELECTOR_CONSTANTS.OverlayNonEditableBorderColor;
        }
        this.m_hudContentDiv.setAttribute('class', classesForElementHUD);
        this.m_overlayDiv.setAttribute('class', classesForOverlay);
	}
};

/*
function:buildSelectorElements - function to generate the elements showing the selectors in HUD

Arguments: None

Return : String: selector elements html structure
*/
EditableSelectors.prototype.buildSelectorElements = function () {
	'use strict';
    //create current selectors structure
    var selectorsStr = '';
    selectorsStr += this.buildSelectorElementsCore(this.m_documentClassesForElement, this.m_liveClassesForElement, '.');
    selectorsStr += this.buildSelectorElementsCore(this.m_documentIdForElement, this.m_liveIdForElement, '#');

    var newSelectorInputStr = '';
    var addButtonStr = '';
    //add newSelector button and input if the element is editable
    if (liveViewObject.isElementEditable(this.m_currentSelectedElement)) {
        newSelectorInputStr = '<' + CONSTANTS.DwDivTagName + ' id="' + SELECTOR_CONSTANTS.NewSelectorTextInputID + '" contenteditable="true">'
                                     + '</' + CONSTANTS.DwDivTagName + '>';
        
        addButtonStr = '<' + CONSTANTS.DwDivTagName + ' id="' + SELECTOR_CONSTANTS.AddNewSelectorButtonID + '" title="'
                                    + this.locStrings.addTooltip + '">'
                                    + SELECTOR_CONSTANTS.AddNewSelectorButtonString + '</' + CONSTANTS.DwDivTagName + '>';
    }
    
    return selectorsStr + newSelectorInputStr + addButtonStr;
};

/*
function:buildSelectorElementsCore - Utility function for building the selector elements

Arguments: originalArrayArg - selectors in document (editable set)
           generatedArrayArg - selectors in live dom
           prefix - prefix to be added ('#' for id and '.' for classes)

Return : String:selectors list properly set with editable/non-editable state
*/
EditableSelectors.prototype.buildSelectorElementsCore = function (originalArrayArg, generatedArrayArg, prefix) {
    'use strict';
    
    var selectorsStr = '';
    var selectorTagStr;
    var selectorElementStr;
    var i;
    //make a copy of the passed arrays so that we do not modify the passed arrays
    var originalArray = originalArrayArg ? originalArrayArg.slice(0) : null;
    var generatedArray = generatedArrayArg ? generatedArrayArg.slice(0) : null;

    //first go through the document selector list and show them as editable with remove button
    if (generatedArray && originalArray && originalArray.length > 0) {
        var removeButtonStr = '';
        if (liveViewObject.isElementEditable(this.m_currentSelectedElement)) {
            removeButtonStr = this.createTagWithClassAndContent(CONSTANTS.DwDivTagName,
                                                                    SELECTOR_CONSTANTS.RemoveSelectorButtonClass,
                                                                    SELECTOR_CONSTANTS.RemoveSelectorButtonString, this.locStrings.removeTooltip);
        }
        for (i = 0; i < originalArray.length; ++i) {
            if (originalArray[i] && originalArray[i].length > 0) {
                var curIndex = generatedArray.indexOf(originalArray[i]);
                if (curIndex !== -1) {
                    selectorTagStr = this.createTagWithClassAndContent(CONSTANTS.DwDivTagName,
                                        SELECTOR_CONSTANTS.SelectorNameClass,
                                        prefix + originalArray[i]);

                    selectorElementStr = this.createTagWithClassAndContent(CONSTANTS.DwDivTagName,
                                            SELECTOR_CONSTANTS.EditableSelectorClass,
                                            selectorTagStr + removeButtonStr);

                    selectorsStr = selectorsStr + selectorElementStr;
                    
                    //remove it from generated array so that it wont be processed again
                    generatedArray.splice(curIndex, 1);
                }
            }
        }
    }

    //now add the remaining in generated array as non-editable
    if (generatedArray && generatedArray.length > 0) {
        for (i = 0; i < generatedArray.length; ++i) {
            if (generatedArray[i] && generatedArray[i].length > 0) {
                selectorTagStr = this.createTagWithClassAndContent(CONSTANTS.DwDivTagName,
                                    SELECTOR_CONSTANTS.SelectorNameClass,
                                    prefix + generatedArray[i]);

                selectorElementStr = this.createTagWithClassAndContent(CONSTANTS.DwDivTagName,
                                            SELECTOR_CONSTANTS.NonEditableSelectorClass,
                                            selectorTagStr);

                selectorsStr = selectorsStr + selectorElementStr;
            }
        }
    }
    
	return selectorsStr;
};

/*
function:buildSelectorElementsCore - Utility function for building a tag string based on input attributes.

Arguments: tagNameStr - name of tag
           tagClassName - class attribute for tag
           tagContentString - contents of the tag

Return : String : properly constructed tag string
*/
EditableSelectors.prototype.createTagWithClassAndContent = function (tagNameStr, tagClassName, tagContentString, title) {
    'use strict';
    var tagString = '<' + tagNameStr;

    if (tagClassName && tagClassName.length > 0) {
        tagString += ' class="' + tagClassName + '"';
    }
       
    if (title && title.length > 0) {
        tagString += ' title="' + title + '"';
    }
    
    tagString += '>';
    
    if (tagContentString && tagContentString.length > 0) {
        tagString += tagContentString;
    }
    
    tagString += '</' + tagNameStr + '>';

    return tagString;
};

/*
function:positionOverLayDiv - Position our overlay div at the specified rect

Arguments: elemRect - rect to be used

Return : none
*/
EditableSelectors.prototype.positionOverLayDiv = function (elemRect) {
    'use strict';
   
    if (this.m_overlayDiv && elemRect) {
        if (elemRect.width <= 0 || elemRect.height <= 0) {
            this.m_overlayDiv.style.display = 'none';
            return;
        }

        this.m_overlayDiv.style.top = elemRect.top + 'px';
        this.m_overlayDiv.style.left = elemRect.left + 'px';
        //we should reduce the width and height to accommodate the border
        //of the overlay div. so reduce 2*border_width    
        this.m_overlayDiv.style.width = (elemRect.width - 2 * CONSTANTS.OverlayBorderWidth) + 'px';
        this.m_overlayDiv.style.height = (elemRect.height - 2 * CONSTANTS.OverlayBorderWidth) + 'px';
        
        this.m_overlayDiv.style.display = 'block';
        var tagname = this.m_currentSelectedElement.tagName.toLowerCase();
        if (tagname !== "img" && tagname !== "video" && tagname !== "audio" && dwObject) {
            var argObj = {};
            argObj.type = CONSTANTS.textHudShown;
            argObj.category = CONSTANTS.startAction;
            argObj.left = window.screenX +  elemRect.left - window.parent.scrollX;
            argObj.top = window.screenY + elemRect.top - window.parent.scrollY;
            argObj.width = elemRect.width;
            argObj.height = elemRect.height;
            argObj.windowscreenX = window.screenX;
            argObj.windowscreenY = window.screenY;
            dwObject.sendTextHudOpenUpdateforNFW(argObj);
        }
    }
};

/*
function:positionHud - positions the HUD based on the rect provided

Arguments: elemRect - rect to be used

Return : none
*/
EditableSelectors.prototype.positionHud = function (elemRect) {
	'use strict';
    
    if (this.m_hudContentDiv && elemRect) {
        if (elemRect.width <= 0 || elemRect.height <= 0) {
            this.destroy();
            return;
        }
        
        if (!parent.document.documentElement) {
            this.destroy();
            return;
        }

		//set our extension to use the entire document space
        var documentWidth = parent.document.width;
        var documentHeight = parent.document.height;
        
        /*
        document.width & document.height are not supported in CEF 1650.
        Should update to use the below code on moving to 1650
        var documentWidth = parent.document.documentElement.offsetWidth;
        var documentHeight = parent.document.documentElement.offsetHeight;
        */

        //in some cases, element stretches beyond the size provided to us by document element
        //so increase the height and width if needed
        if (elemRect.left + elemRect.width > documentWidth) {
            documentWidth = elemRect.left + elemRect.width;
        }
        
        if (elemRect.top + elemRect.height > documentHeight) {
            documentHeight = elemRect.top + elemRect.height;
        }
        
        if(documentHeight < CONSTANTS.ExtensionMinHeight) {
            documentHeight = CONSTANTS.ExtensionMinHeight;
        }
           
        liveViewObject.positionExtensionById(CONSTANTS.ExtensionID, 0, 0, documentWidth, documentHeight);
        
        //position border overlay
        this.positionOverLayDiv(elemRect);
        
        //display HUD and then position it
		this.m_hudContentDiv.style.display = 'block';
        
        //if there is no space at top, keep the HUD at bottom
		var hudTop = elemRect.top - this.m_hudContentDiv.offsetHeight;
        if (hudTop < 0) {
			this.m_hudIsOnTop = false;
			hudTop = elemRect.top + elemRect.height;
            //if the hud exceeds the document height, position it just at top
            if (hudTop + this.m_hudContentDiv.offsetHeight > documentHeight) {
                hudTop = elemRect.top;
            }
		} else {
            this.m_hudIsOnTop = true;
        }
            
        //set our dimensions same as parent document
        document.body.width = documentWidth;
        document.body.height = documentHeight;
        
        //now calculate the right based on the parent document rect
        //if there is no space on right, move left
		var hudLeft = elemRect.left;
		if (hudLeft + this.m_hudContentDiv.offsetWidth > documentWidth) {
			hudLeft = elemRect.left + elemRect.width - this.m_hudContentDiv.offsetWidth;
        }
        
        this.m_hudContentDiv.style.top = hudTop + 'px';
		this.m_hudContentDiv.style.left = hudLeft + 'px';
	}
    if (dwObject) {
        dwObject.sendNFWUpdateToBridge(CONSTANTS.eshShown, CONSTANTS.endActionCategory);
    }
   
};

/*
function:ClickHandler - Mouse click handler for ESH elements

Arguments: event - event passed from browser

Return : none
*/
EditableSelectors.prototype.ClickHandler = function (event) {
    'use strict';
    
    if (!this.m_currentSelectedElement) {
        return;
    }
    
    var eventTarget = event.target;
	var targetId	= eventTarget.getAttribute('id');
    var targetClass = eventTarget.getAttribute('class');
	
	//if the click is on a selector, expand it
    if (eventTarget.id === CONSTANTS.hudEntryID) {
        this.toggleAuxHudState(false);
        event.stopPropagation();
        event.preventDefault();
    } else if (eventTarget && targetClass && targetClass === SELECTOR_CONSTANTS.SelectorNameClass) {
        var newClass = targetClass + ' ' + SELECTOR_CONSTANTS.ExpandedSelectorClass;
        eventTarget.setAttribute('class', newClass);
        this.repositionHUD();
    } else if (eventTarget && eventTarget.id === SELECTOR_CONSTANTS.AddNewSelectorButtonID) {
        //show the input field and focus
        var inputElement = document.getElementById(SELECTOR_CONSTANTS.NewSelectorTextInputID);
        if (inputElement) {
			inputElement.innerHTML = '';
			inputElement.style.display = 'inline-block';
			inputElement.focus();

            //attach the autocomplete widget to input
            //by default, suggestions come below the input field. If there is no space below, then the 
            //suggestions should come above the input            
            var suggestionListPositionMy = "left top";
            var suggestionListPositionAt = "left bottom";
            var hudRect = this.m_hudContentDiv.getBoundingClientRect();
            if (hudRect.bottom + window.scrollY + CONSTANTS.SuggestionsListMaxHeight > document.body.height) {
                suggestionListPositionMy = "left bottom";
                suggestionListPositionAt = "left top";
            }
            $(SELECTOR_CONSTANTS.NewSelectorTextInputJQueryID).autocomplete({select: this.newSelectorItemSelected.bind(this)},
                {focus: this.focusChangedInSuggestions.bind(this)},
                {change: this.focusLostFromNewSelectorInput.bind(this)}, {position: { my: suggestionListPositionMy, at: suggestionListPositionAt, collision: "fit none" }});

            //handle Enter key in the input
            $(SELECTOR_CONSTANTS.NewSelectorTextInputJQueryID).keydown(function (e) {
                if (e.keyCode === CONSTANTS.EnterKeyCode) {
                    e.stopPropagation();
                    e.preventDefault();
                    this.newSelectorItemSelected(null, null);
                } else {
                    //see whether typing has increased the width of HUD to cross the boundaries
                    //if it exceeds the extension width, then scrollbars will appear.
                    //so increase extension width if needed
                    var hudRect = this.m_hudContentDiv.getBoundingClientRect();
                    if (hudRect.right > document.body.width) {
                        liveViewObject.positionExtensionById(CONSTANTS.ExtensionID, 0, 0, hudRect.right, document.body.height);
                        document.body.width = hudRect.right;
                    }
                }
            }.bind(this));

            //populate the autocomplete with all the selectors available in the document
            var callbackFunction = function (allSelectors) {
                this.attachSelectorHints(allSelectors);
            }.bind(this);

			dwObject.getDWDocumentClassesAndIds(callbackFunction);

            //Hud height might have changed
            //do the positioning again
            this.repositionHUD();
            this.hideAuxHud(true, true);
		}
    } else if (eventTarget && targetClass && targetClass === SELECTOR_CONSTANTS.RemoveSelectorButtonClass) {
        var selectorNameElement = eventTarget.previousSibling;

        if (selectorNameElement) {
            var attrToRemove = selectorNameElement.innerHTML;
            this.removeSelector(attrToRemove);
            //clear browser dom modifications we have done, so that undo will go to DW document
            //and revert this operaion
            liveViewObject.clearBrowserUndoRedos();
        }
    }
};

/*
function:attachSelectorHints - callback from DW with all the selectors in the document. Attach it to autocomplete widget

Arguments: allSelectors - list of selectors in the document

Return : none
*/
EditableSelectors.prototype.attachSelectorHints = function (allSelectors) {
    'use strict';
    var availableSelectors = allSelectors;
    //attach to autocomplete
    $(SELECTOR_CONSTANTS.NewSelectorTextInputJQueryID).autocomplete('option', 'source', function (request, response) {
        var curText = request.term;
        var terms = curText.split(' ');
        
        //remove selectors which are currently applied on the element and which are already present in the input
        var lastTerm = terms.pop();
        var newSelectors = this.getListOfSelectorsForAutoComplete(availableSelectors, terms);
        
        //get the last term in the input field, which is the input for suggestions        
        response($.ui.autocomplete.filter(newSelectors, lastTerm));
    }.bind(this));
};

/*
function:getListOfSelectorsForAutoComplete - returns the list of selectors that can be applied to the current element.
    removes the selectors currently applied on the element from the passed list

Arguments: allSelectors - list of selectors in the document
           removeItems - list of items to be removed other than the applied ones

Return : newSelectorsAvailable = allSelectors - currently_Applied_Selectors
*/
EditableSelectors.prototype.getListOfSelectorsForAutoComplete = function (allSelectors, removeItems) {
    'use strict';
    
    var newSelectorsAvailable = null;
    var i;
    var index;

    //allselectors is the list of all selectors in the document.
    if (allSelectors && allSelectors.length > 0) {
        //assigning array directly will be a reference
        //so make a copy
        newSelectorsAvailable = allSelectors.slice(0);

        //remove the classes currently applied on the element.
        if (this.m_documentClassesForElement) {
            for (i = 0; i < this.m_documentClassesForElement.length; ++i) {
                index = newSelectorsAvailable.indexOf('.' + this.m_documentClassesForElement[i]);
                if (index !== -1) {
                    newSelectorsAvailable.splice(index, 1);
                }
            }
        }

        //similarly remove the id currently applied on the element.
        if (this.m_documentIdForElement) {
            for (i = 0; i < this.m_documentIdForElement.length; ++i) {
                index = newSelectorsAvailable.indexOf('#' + this.m_documentIdForElement[i]);
                if (index !== -1) {
                    newSelectorsAvailable.splice(index, 1);
                }
            }
        }
        
        //now remove everything from remove items
        if (removeItems) {
            for (i = 0; i < removeItems.length; ++i) {
                var curItem = removeItems[i];
                var itemIndex = -1;
                //if it is specified as class or ID, then use it
                //or take this as a class
                if (curItem.charAt(0) === '#' || curItem.charAt(0) === '.') {
                    itemIndex = newSelectorsAvailable.indexOf(curItem);
                } else {
                    itemIndex = newSelectorsAvailable.indexOf('.' + curItem);
                }
                
                //if found, remove from list
                if (itemIndex !== -1) {
                    newSelectorsAvailable.splice(itemIndex, 1);
                }
            }
        }
    }

    return newSelectorsAvailable;
};

/*
JQuery AutoComplete widget callback functions
*/

/*
Handles cases when focus changes in autocomplete suggestions
*/
EditableSelectors.prototype.focusChangedInSuggestions = function (event, ui) {
	'use strict';
    //default is to set the focused node content to input box,
    //we do not need to update the input field since it sets the cursor 
    //to the beginning of text in input filed
    if (event) {
        event.preventDefault();
    }
    
    //auto populate input box only if the focus changed through keyboard
    //original even is actually nested inside jQuery event
    var origEvent = event;
    while (origEvent && origEvent.originalEvent) {
        origEvent = origEvent.originalEvent;
    }
    if (origEvent && origEvent.type !== "keydown") {
        return;
    }

    //let us set the suggestion text in intupt field and set the cursor properly
    var inputElem = document.getElementById(SELECTOR_CONSTANTS.NewSelectorTextInputID);
    if (ui && ui.item && inputElem) {
        //since we allow multiple selectors, we should append the current suggestion
        //to the current list of selectors in the input field
        var itemString = ui.item.value;
        
        var curValue = inputElem.innerHTML;
        //we should replace only the last term with the suggestion item
        var lastTermStartindex = curValue.lastIndexOf(" ");
        if (lastTermStartindex !== -1) {
            curValue = curValue.substr(0, lastTermStartindex);
        } else {
            curValue = "";
        }
        //add a space as separator
        if (curValue.length > 0) {
            curValue += " ";
        }
        //now set the valu in the filed
        inputElem.innerHTML = curValue + itemString;
        
        //set the document selection to set cursor at proper position
        if (itemString && inputElem.childNodes[0]) {
            var range = document.createRange();
            var sel = window.getSelection();
            range.setStart(inputElem.childNodes[0], inputElem.innerHTML.length);
            range.collapse(true);
            sel.removeAllRanges();
            sel.addRange(range);
        }
    }
};

/*
Handles cases when autocomplete box loses focus.
*/
EditableSelectors.prototype.focusLostFromNewSelectorInput = function (event, ui) {
	'use strict';
    
    //remove autocomplete
    $(SELECTOR_CONSTANTS.NewSelectorTextInputJQueryID).autocomplete('destroy');
    $(SELECTOR_CONSTANTS.NewSelectorTextInputJQueryID).unbind("keydown");
    //if the focus is not in the input field, hide it
    var inputElem = document.getElementById(SELECTOR_CONSTANTS.NewSelectorTextInputID);
	if (inputElem) {
		inputElem.style.display = 'none';
		inputElem.innerHTML = '';
        inputElem.blur();
	}

    //Hud height might have changed
    //do the positioning again
    this.repositionHUD();
    
    //clear browser dom modifications we have done, so that undo will go to DW document
    liveViewObject.clearBrowserUndoRedos();
};

/*
Handles selection inside the autocomplete popup window.
*/
EditableSelectors.prototype.newSelectorItemSelected = function (event, ui) {
	'use strict';
    
    //if this event was triggered due to a selection using keyboard, 
    //then the input file d will have the exact value to be applied.
    //if it was trigeered by a mouse click, then we should do additional stuff
    //to get the value to be applied
    var selectedUsingKeyBoard = true;
    //we do not need to update the input field
    if (event) {
        event.preventDefault();
        //original event is included at some depth
        var origEvent = event;
        while (origEvent && origEvent.originalEvent) {
            origEvent = origEvent.originalEvent;
        }
        if (origEvent && origEvent.type !== "keydown") {
            selectedUsingKeyBoard = false;
        }
    }
    
    var valueSubmitted = "";
    var inputElem = document.getElementById(SELECTOR_CONSTANTS.NewSelectorTextInputID);
    //if the selection was using mouse click, then we should replace the 
    //last term in the input filed with the suggestion item value
    if (inputElem && !selectedUsingKeyBoard && ui && ui.item && ui.item.value) {
        var itemString = ui.item.value;
        var curValue = inputElem.innerHTML;
        if (curValue && curValue.length > 0) {
            var lastTermStartindex = curValue.lastIndexOf(" ");
            if (lastTermStartindex !== -1) {
                curValue = curValue.substr(0, lastTermStartindex);
            } else {
                curValue = "";
            }
            if (curValue.length > 0) {
                curValue += " ";
            }
        } else {
            curValue = "";
        }
        valueSubmitted = curValue + itemString;
    } else {
        valueSubmitted = inputElem.innerHTML;
    }
    
    //cleaup input field
    this.focusLostFromNewSelectorInput(event, ui);
    
    //apply new selectors
    this.applyNewSelectors(valueSubmitted.trim());
    
    //clear browser dom modifications we have done, so that undo will go to DW document
    //and revert this operaion
    liveViewObject.clearBrowserUndoRedos();
};

/*
function:applyNewSelector - Function to add the set of new selectors to the current element. Adds to both live dom and DW document

Arguments: newSelector - new selectors to be added

Return : none
*/
EditableSelectors.prototype.applyNewSelectors = function (newSelector) {
    'use strict';
    if (this.m_currentSelectedElement && newSelector && newSelector.length > 0) {
        var selectorsArray = newSelector.split(' ');
        var idToApply = "";
        var classesToApply = [];
        var i;
        //get ID to apply and classes to apply
        //if multiple ids are provided, then we will apply the last one.
        //if selector starts with '#' then its an id, else it is a class
        for (i = 0; i < selectorsArray.length; i++) {
            var currentSelector = selectorsArray[i];
            if (currentSelector.charAt(0) === '#') {
                var currentID = currentSelector.slice(1);
                if (currentID.length > 0) {
                    idToApply = currentID;
                }
            } else {
                var currentClass = (currentSelector.charAt(0) === '.' ? currentSelector.slice(1) : currentSelector);
                if (currentClass.length > 0) {
                    classesToApply.push(currentClass);
                }
            }
        }
        
        //apply the ID
        if (idToApply.length > 0) {
            //apply to current element
            this.m_currentSelectedElement.setAttribute('id', idToApply);
            //update our data
            this.m_documentIdForElement = idToApply.split(' ');
            //propogate to Document
            dwObject.updateDWDocumentElementAttr(this.m_currentSelectedElement, 'id', idToApply);
        }
        
        //apply classes
        if (classesToApply.length > 0) {
            var curClasses = this.m_currentSelectedElement.getAttribute('class');
            if (curClasses && curClasses.length > 0) {
                curClasses = curClasses + ' ' + classesToApply.join(' ');
            } else {
                curClasses = classesToApply.join(' ');
            }
                
            //apply to current element
            this.m_currentSelectedElement.setAttribute('class', curClasses);

            if (this.m_documentClassesForElement) {
                this.m_documentClassesForElement = this.m_documentClassesForElement.concat(classesToApply);
            } else {
                this.m_documentClassesForElement = classesToApply;
            }

            //propogate to Document
            var newClasses = this.m_documentClassesForElement.join(' ');
            dwObject.updateDWDocumentElementAttr(this.m_currentSelectedElement, 'class', newClasses);
        }
        
        //add headlights data
        var idApplied = idToApply.length > 0;
        var numClassesApplied = classesToApply.length;
        if (idApplied || numClassesApplied > 0) {
            dwObject.logHeadlightsData(DW_ESH_HEADLIGHTS.ELV_ESH, DW_ESH_HEADLIGHTS.ESH_ADD);
            //log if multiple selectors were added
            if ((idApplied && numClassesApplied > 0) || numClassesApplied > 1) {
                dwObject.logHeadlightsData(DW_ESH_HEADLIGHTS.ELV_ESH, DW_ESH_HEADLIGHTS.ESH_MULTIADD);
            }
        }
        
        // After the selector has been applied, send nfw event
        if (dwObject) {
            dwObject.sendNFWUpdateToBridge(CONSTANTS.eshAddSelector, CONSTANTS.startAction);
        }
        //update the HUD and bring the element back into view
        //incase it has moved
        this.createElementHud();
    }
};

/*
function:removeSelector - Function to remove a new selector from the current element. removes from both live dom and DW document

Arguments: newSelector - selector to be removed

Return : none
*/
EditableSelectors.prototype.removeSelector = function (attrValToRemove) {

    'use strict';
    var theDocumentAttrValList = null;
    var theLiveAttrValList = null;
    var theAttrName = null;

    if (!(attrValToRemove && attrValToRemove.length > 0)) {
        return;
    }

    //check whether it is a class or id
    if (attrValToRemove.charAt(0) === '.') {
        theDocumentAttrValList = this.m_documentClassesForElement;
        theLiveAttrValList = this.m_liveClassesForElement;
        theAttrName = 'class';
    } else if (attrValToRemove.charAt(0) === '#') {
        theDocumentAttrValList = this.m_documentIdForElement;
        theLiveAttrValList = this.m_liveIdForElement;
        theAttrName = 'id';
    } else {
        return;
    }

    if (theAttrName && theDocumentAttrValList && theDocumentAttrValList.length > 0 && this.m_currentSelectedElement) {
        //remove first char '.' or '#'
        attrValToRemove = attrValToRemove.slice(1);
        var valIndex = theDocumentAttrValList.indexOf(attrValToRemove);
        if (valIndex !== -1) {
            // Update current element
            var liveValIndex = theLiveAttrValList.indexOf(attrValToRemove);
            if (liveValIndex !== -1) {
                theLiveAttrValList.splice(liveValIndex, 1);
                var liveAttrValStr = theLiveAttrValList.join(' ');
                if (liveAttrValStr && liveAttrValStr.length > 0) {
                    this.m_currentSelectedElement.setAttribute(theAttrName, liveAttrValStr);
                } else {
                    this.m_currentSelectedElement.removeAttribute(theAttrName);
                }
            }
            
            //update document
            theDocumentAttrValList.splice(valIndex, 1);
            var newDocumentAttrValStr = theDocumentAttrValList.join(' ');
            dwObject.updateDWDocumentElementAttr(this.m_currentSelectedElement, theAttrName, newDocumentAttrValStr);
            
            //log headlights
            dwObject.logHeadlightsData(DW_ESH_HEADLIGHTS.ELV_ESH, DW_ESH_HEADLIGHTS.ESH_DELETE);
            
            //update the HUD and bring the element back into view
            //incase it has moved
            this.createElementHud();
        }
    }
};

/*
function:repositionHUD - updates the position of HUD

Arguments: none

Return : none
*/
EditableSelectors.prototype.repositionHUD = function () {
    'use strict';
    if (this.m_currentSelectedElement) {
        var elemRect = liveViewObject.getElementRect(this.m_currentSelectedElement);
        this.positionHud(elemRect);
    }
};

/*
function:scrollHUDIntoViewIfNeeded - Function to bring HUD back into view if it the element
                                     is outside the current viewable area
Arguments: none

Return : none
*/
EditableSelectors.prototype.scrollHUDIntoViewIfNeeded = function () {
    'use strict';
    if (!this.m_currentSelectedElement) {
        return;
    }

    //get where our current element is
    var elemRect = liveViewObject.getElementRect(this.m_currentSelectedElement);
    if (elemRect.width <= 0 || elemRect.height <= 0) {
        return;
    }

    //Get the current viewable window cordinates
    var windowTop = parent.window.scrollY;
    var windowBottom = windowTop + parent.window.innerHeight;
    var windowLeft = parent.window.scrollX;
    var windowRight = windowLeft + parent.window.innerWidth;

    //if our HUD is not within the viewable window, lets scroll
    if (elemRect.left > windowRight || elemRect.left + elemRect.width < windowLeft ||
            elemRect.top > windowBottom || elemRect.top + elemRect.height < windowTop) {
        this.m_currentSelectedElement.scrollIntoView();
        //if our HUD is on top of element, then lets scroll that much so that HUD will be visible
        if (this.m_hudIsOnTop) {
            parent.window.scrollBy(0, this.m_hudContentDiv.offsetHeight * -1);
        }
    }
};

/*
function:initEditableSelectors - Entry point for Editable Selectors. Set on body Onload in HTML

Arguments: none

Return : none
*/
var initEditableSelectors = function () {
    'use strict';
    window.editableSelectorsObj = new EditableSelectors();
    window.editableSelectorsObj.initialize();
};
